export class GameConfig {
    constructor(configSchema, customStyles = {}) {
        this._schema = configSchema;
        this._bindings = new Map();
        this._values = {};
        this._customStyles = customStyles;
        
        // Initialize values from schema or local storage
        Object.entries(this._schema).forEach(([key, settings]) => {
            // Try to get value from local storage first
            const storedValue = localStorage.getItem(`gameConfig_${key}`);
            this._values[key] = storedValue !== null ? JSON.parse(storedValue) : settings.default;
            
            // Define getters and setters for each property
            Object.defineProperty(this, key, {
                get: () => this._values[key],
                set: (value) => {
                    const parsedValue = this._parseValue(value, settings.type);
                    this._values[key] = parsedValue;
                    this._updateBindings(key, parsedValue);
                    // Save to local storage
                    localStorage.setItem(`gameConfig_${key}`, JSON.stringify(parsedValue));
                }
            });
        });
    }

    // Bind to existing DOM element
    bindToElement(propertyName, element) {
        if (!this._schema[propertyName]) {
            console.error(`Property ${propertyName} not found in schema`);
            return;
        }

        if (!this._bindings.has(propertyName)) {
            this._bindings.set(propertyName, new Set());
        }
        
        this._bindings.get(propertyName).add(element);
        
        // Set up event listener based on element type
        if (element.type === 'radio') {
            element.addEventListener('change', () => {
                if (element.checked) {
                    this[propertyName] = element.value;
                }
            });
            // Set initial state
            element.checked = element.value === String(this[propertyName]);
        } else if (element.type === 'checkbox') {
            element.addEventListener('change', () => {
                this[propertyName] = element.checked;
            });
            element.checked = this[propertyName];
        } else if (element.type === 'range') {
            element.addEventListener('input', () => {
                this[propertyName] = element.value;
            });
            element.value = this[propertyName];
        } else {
            element.addEventListener('input', () => {
                this[propertyName] = element.value;
            });
            element.value = this[propertyName];
        }
    }

    // Bind to multiple radio buttons
    bindRadioGroup(propertyName, elements) {
        elements.forEach(element => this.bindToElement(propertyName, element));
    }

    // Generate form based on schema
    generateForm(containerId = null) {
        // Create main container
        const container = document.createElement('div');
        container.className = 'game-config-container';

        // Create toggle button
        const toggleButton = document.createElement('button');
        toggleButton.className = 'config-toggle-btn';
        toggleButton.textContent = 'Show Config';
        container.appendChild(toggleButton);

        // Create form
        const form = document.createElement('form');
        form.className = 'game-config-form';
        form.style.display = 'none';  // Initially hidden
        container.appendChild(form);

        // Toggle button functionality
        toggleButton.onclick = (e) => {
            e.preventDefault();
            const isHidden = form.style.display === 'none';
            form.style.display = isHidden ? 'block' : 'none';
            toggleButton.textContent = isHidden ? 'Hide Config' : 'Show Config';
        };

        Object.entries(this._schema).forEach(([key, settings]) => {
            const formGroup = document.createElement('div');
            formGroup.className = 'form-group';

            // Create label
            const label = document.createElement('label');
            label.textContent = settings.label || this._formatLabel(key);
            
            if(settings.type !== 'checkbox') {
                formGroup.appendChild(label);
            }

            // Create input based on type
            switch (settings.type) {
                case 'radio':
                    settings.options.forEach(option => {
                        const radioDiv = document.createElement('div');
                        radioDiv.className = 'radio-option';
                        
                        const radio = document.createElement('input');
                        radio.type = 'radio';
                        radio.name = key;
                        radio.value = option.value;
                        radio.id = `${key}-${option.value}`;
                        
                        const radioLabel = document.createElement('label');
                        radioLabel.textContent = option.label;
                        radioLabel.htmlFor = radio.id;
                        
                        radioDiv.appendChild(radio);
                        radioDiv.appendChild(radioLabel);
                        formGroup.appendChild(radioDiv);
                        
                        this.bindToElement(key, radio);
                    });
                    break;

                case 'checkbox':
                    const checkbox = document.createElement('input');
                    checkbox.type = 'checkbox';
                    checkbox.id = key;
                    this.bindToElement(key, checkbox);
                    formGroup.appendChild(checkbox);
                    formGroup.appendChild(label);
                    break

                case 'range':
                    const sliderContainer = document.createElement('div');
                    sliderContainer.className = 'slider-container';
                    
                    const slider = document.createElement('input');
                    slider.type = 'range';
                    slider.min = settings.min || 0;
                    slider.max = settings.max || 100;
                    slider.step = settings.step || 1;
                    slider.id = key;
                    
                    const valueDisplay = document.createElement('span');
                    valueDisplay.className = 'slider-value';
                    valueDisplay.textContent = this[key];
                    
                    slider.addEventListener('input', () => {
                        valueDisplay.textContent = slider.value;
                    });
                    
                    sliderContainer.appendChild(slider);
                    sliderContainer.appendChild(valueDisplay);
                    formGroup.appendChild(sliderContainer);
                    
                    this.bindToElement(key, slider);
                    break;

                default: // text or number
                    const input = document.createElement('input');
                    input.type = settings.type;
                    if (settings.type === 'number') {
                        input.min = settings.min;
                        input.max = settings.max;
                        input.step = settings.step || 1;
                        // Enable spinner buttons
                        input.className = 'number-input';
                    }
                    input.id = key;
                    this.bindToElement(key, input);
                    formGroup.appendChild(input);
            }

            form.appendChild(formGroup);
        });

        // Add basic styling
        const style = document.createElement('style');
        style.textContent = `
            .game-config-form {
                max-width: 400px;
                padding: 20px;
                background: #f8f9fa;
                border-radius: 8px;
                box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            }
            .form-group {
                margin-bottom: 20px;
            }
            .form-group label {
                display: block;
                margin-bottom: 8px;
                font-weight: 600;
                color: #2c3e50;
                font-size: 14px;
            }
            .radio-option {
                margin: 8px 0;
                display: flex;
                align-items: center;
                gap: 8px;
            }
            .radio-option input[type="radio"] {
                margin: 0;
            }
            .radio-option label {
                margin: 0;
                font-weight: normal;
            }
            .slider-container {
                display: flex;
                align-items: center;
                gap: 12px;
                padding: 4px 0;
            }
            .slider-value {
                min-width: 45px;
                padding: 4px 8px;
                background: #e9ecef;
                border-radius: 4px;
                text-align: center;
                font-size: 14px;
            }
            input[type="number"],
            input[type="text"] {
                padding: 8px 12px;
                border: 1px solid #ced4da;
                border-radius: 4px;
                font-size: 14px;
                transition: border-color 0.2s;
            }
            input[type="number"]:focus,
            input[type="text"]:focus {
                outline: none;
                border-color: #4a90e2;
                box-shadow: 0 0 0 3px rgba(74,144,226,0.1);
            }
            input[type="range"] {
                -webkit-appearance: none;
                width: 100%;
                height: 6px;
                background: #dee2e6;
                border-radius: 3px;
                margin: 10px 0;
            }
            input[type="range"]::-webkit-slider-thumb {
                -webkit-appearance: none;
                width: 18px;
                height: 18px;
                background: #4a90e2;
                border-radius: 50%;
                cursor: pointer;
                margin-top: -6px;
                box-shadow: 0 1px 3px rgba(0,0,0,0.2);
                transition: background 0.2s;
            }
            input[type="range"]::-webkit-slider-runnable-track {
                width: 100%;
                height: 6px;
                background: #dee2e6;
                border-radius: 3px;
            }
            input[type="range"]:focus {
                outline: none;
            }
            input[type="range"]::-webkit-slider-thumb:hover {
                background: #357abd;
            }
            input[type="checkbox"] {
                width: 16px;
                height: 16px;
                margin-right: 8px;
                cursor: pointer;
            }
            .radio-option input[type="radio"] {
                width: 16px;
                height: 16px;
                cursor: pointer;
            }
            /* Better slider styling */
            input[type="range"] {
                --range-progress: 50%;
                -webkit-appearance: none;
                appearance: none;
                width: 100%;
                height: 6px;
                border-radius: 3px;
                background: linear-gradient(to right, #4a90e2 var(--range-progress), #dee2e6 0%);
                transition: background 0.3s;
            }
            /* Ensure the track goes edge to edge */
            input[type="range"]::-webkit-slider-runnable-track {
                -webkit-appearance: none;
                appearance: none;
            }
            /* Style the thumb to extend properly */
            input[type="range"]::-webkit-slider-thumb {
                -webkit-appearance: none;
                appearance: none;
                width: 18px;
                height: 18px;
                border-radius: 50%;
                background: #4a90e2;
                cursor: pointer;
                border: none;
                margin-top: -6px;
                box-shadow: 0 1px 3px rgba(0,0,0,0.2);
                transition: background 0.3s;
            }
            /* Number input styling */
            input[type="number"] {
                -moz-appearance: textfield;
                padding-right: 15px !important; /* Make room for smaller arrows */
            }
            input[type="number"]::-webkit-outer-spin-button,
            input[type="number"]::-webkit-inner-spin-button {
                -webkit-appearance: none;
                background: #f8f9fa url('data:image/svg+xml;utf8,<svg width="10" height="20" xmlns="http://www.w3.org/2000/svg"><g fill="%234a90e2"><path d="M5 0l5 5H0z"/><path d="M5 20l5-5H0z"/></g></svg>') no-repeat center right;
                width: 15px;
                height: 100%;
                opacity: 1;
                cursor: pointer;
                margin-right: 2px;
            }
            /* Checkbox styling */
            .checkbox-container {
                display: flex;
                align-items: center;
                gap: 0.5rem;
            }
            .checkbox-container input[type="checkbox"] {
                width: 20px;
                height: 20px;
                accent-color: #4a90e2;
                cursor: pointer;
            }
            .checkbox-container label {
                font-size: 14px;
                color: #2c3e50;
                cursor: pointer;
                user-select: none;
            }
            .checkbox-container input[type="checkbox"]:hover {
                box-shadow: 0 0 5px rgba(74, 144, 226, 0.5);
            }
            .checkbox-container input[type="checkbox"]:focus {
                outline: 2px solid rgba(74, 144, 226, 0.8);
            }
            .checkbox-container label:hover {
                color: #357abd;
            }
        `;
        document.head.appendChild(style);

        // KK Added: handle custom styles here
        if(this._customStyles) {
            const customStyles = document.createElement('style');
            customStyles.textContent = this._customStyles;
            document.head.appendChild(customStyles);
        }

        // If containerId provided, append to that element
        if (containerId) {
            const targetContainer = document.getElementById(containerId);
            if (targetContainer) {
                targetContainer.appendChild(container);
            }
        }

        return container;
    }

    // Helper to format labels
    _formatLabel(key) {
        return key
            .replace(/([A-Z])/g, ' $1')
            .replace(/^./, str => str.toUpperCase());
    }

    // Parse value based on type
    _parseValue(value, type) {
        switch (type) {
            case 'number':
            case 'range':
                return Number(value);
            case 'checkbox':
                return Boolean(value);
            default:
                return value;
        }
    }

    // Update all bindings for a property
    _updateBindings(propertyName, value) {
        const bindings = this._bindings.get(propertyName);
        if (!bindings) return;

        bindings.forEach(element => {
            if (element.type === 'radio') {
                element.checked = element.value === String(value);
            } else if (element.type === 'checkbox') {
                element.checked = value;
            } else {
                element.value = value;
            }
        });
    }

    // Export configuration
    exportConfig() {
        return { ...this._values };
    }

    // Import configuration
    importConfig(config) {
        Object.entries(config).forEach(([key, value]) => {
            if (this._schema[key]) {
                this[key] = value;
            }
        });
    }
}