export class JSONFormatter {
    constructor(options = {}) {
      this.defaultOptions = {
        indentSpaces: 2,
        backgroundColor: '#f5f5f5',
        padding: '10px',
        borderRadius: '4px',
        fontFamily: 'monospace',
        fontSize: '14px',
        maxHeight: null,
        theme: {
          string: '#22863a',
          number: '#005cc5',
          boolean: '#005cc5',
          null: '#5a5a5a',
          key: '#24292e'
        }
      };
  
      this.options = { ...this.defaultOptions, ...options };
    }
  
    format(jsonData) {
      try {
        // Convert to string if object is passed
        const jsonString = typeof jsonData === 'string' 
          ? jsonData 
          : JSON.stringify(jsonData, null, this.options.indentSpaces);
  
        return this.createFormattedElement(jsonString);
      } catch (error) {
        return this.createErrorElement(error);
      }
    }
  
    formatAndAppend(jsonData, targetElement) {
      const formattedElement = this.format(jsonData);
      targetElement.appendChild(formattedElement);
      return formattedElement;
    }
  
    createFormattedElement(jsonString) {
      const pre = document.createElement('pre');
      const code = document.createElement('code');
      
      // Apply styles
      Object.assign(pre.style, {
        backgroundColor: this.options.backgroundColor,
        padding: this.options.padding,
        borderRadius: this.options.borderRadius,
        fontFamily: this.options.fontFamily,
        fontSize: this.options.fontSize,
        overflow: 'auto',
        margin: '0'
      });
  
      if (this.options.maxHeight) {
        pre.style.maxHeight = this.options.maxHeight;
      }
  
      // Syntax highlighting
      const highlighted = this.applySyntaxHighlighting(jsonString);
      code.innerHTML = highlighted;
      
      pre.appendChild(code);
      return pre;
    }
  
    createErrorElement(error) {
      const errorElement = document.createElement('div');
      errorElement.style.color = '#dc3545';
      errorElement.style.padding = '10px';
      errorElement.textContent = `Error formatting JSON: ${error.message}`;
      return errorElement;
    }
  
    applySyntaxHighlighting(jsonString) {
      return jsonString
        .replace(/&/g, '&amp;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;')
        .replace(/("(\\u[a-zA-Z0-9]{4}|\\[^u]|[^\\"])*"(\s*:)?|\b(true|false|null)\b|-?\d+(?:\.\d*)?(?:[eE][+\-]?\d+)?)/g, 
          (match) => {
            let color = this.options.theme.number; // number
            if (/^"/.test(match)) {
              color = /:$/.test(match) 
                ? this.options.theme.key   // key
                : this.options.theme.string; // string
            } else if (/true|false/.test(match)) {
              color = this.options.theme.boolean; // boolean
            } else if (/null/.test(match)) {
              color = this.options.theme.null; // null
            }
            return `<span style="color: ${color}">${match}</span>`;
        });
    }
  
    updateOptions(newOptions) {
      this.options = { ...this.options, ...newOptions };
    }
  }
  
  // Usage examples
  const sampleData = {
    name: "John Doe",
    age: 30,
    isActive: true,
    interests: ["programming", "reading"],
    address: {
      street: "123 Main St",
      city: "Boston",
      country: "USA"
    },
    metadata: null
  };

  /*
  
  // Basic usage
  const formatter = new JSONFormatter();
  const container = document.getElementById('json-container');
  formatter.formatAndAppend(sampleData, container);
  
  // Custom options usage
  const customFormatter = new JSONFormatter({
    indentSpaces: 4,
    backgroundColor: '#1e1e1e',
    padding: '20px',
    theme: {
      string: '#ce9178',
      number: '#b5cea8',
      boolean: '#569cd6',
      null: '#569cd6',
      key: '#9cdcfe'
    }
  });
  
  const darkContainer = document.getElementById('dark-json-container');
  customFormatter.formatAndAppend(sampleData, darkContainer);
  
  // Updating options dynamically
  formatter.updateOptions({
    maxHeight: '300px',
    fontSize: '16px'
  });
  
  // Format string JSON
  const stringJSON = '{"name": "Jane", "age": 25}';
  formatter.formatAndAppend(stringJSON, container);
  
  // Error handling example
  const invalidJSON = '{name: invalid}';
  formatter.formatAndAppend(invalidJSON, container);
  
  // Export for module usage
  export default JSONFormatter;

  */