import { 
	USE_PLAYER_NAMES,
	DEBUG_DISPLAY_ALL_TEMPERAMENTS,
	DEBUG_NFC_DISABLED,
	GameStates, 
    GameStateMessages, 
    PlayerStates,
    PlayerStateMessages,
    AnimalInteractions,
    AnimalInteractionMessages,
    GameEndingConditions,
    GameEndingMessages
} from './GameConstants.js';

import { Animals } from './animals.js';
import { Players } from './players.js';
import { NFC } from './nfc.js';
import { Treats } from './treats.js';
import { NameSelector } from './NameSelector.js';
import { GameConfig } from './GameConfig.js';
import { JSONFormatter } from './JSONFormatter.js';


class AnimalFriendsGame {

	// global config defaults
	maxPlayers = 4;  // default overidden by config
	treatsPerPlayer = 2;  // default overidden by config
	treatsTotalValuePerPlayer = 250;  // default overidden by config
	animalsTotalSnarlish = 1; // default overidden by config

	//currentPlayer = null;
	totalPlayers = 2;
	friendThreshold = 100;
	//totalSnarlishAnimals = 1; DEPRECIATED
	bitesToEndGame = 3;
	showExtendedInfo = true;

	// DOM IDs
	upperMessageDOM = 'upperMessageArea';
	treatDisplayDOM = 'playerTreats';
	treatsContainer = document.getElementById("playerTreats");
	
	gameState = 0;
	playerState = 0;

	// store an id for any spawning animations we create
	spawnerId = null;

	// our collections
	treats = null;
	animals = null;
	nfc = null;
	players = null;

	constructor(parms) {

		// config scheme for our game configuration
		const configSchema = {
			maxPlayers: {
				type: 'number',
				default: this.maxPlayers,
				min: 1,
				max: 10
			},
			treatsPerPlayer: {
				type: 'range',
				default: this.treatsPerPlayer,
				min: 1,
				max: 20,
				step: 1,
				label: 'Treats Per Player'				
			},			
			treatsTotalValuePerPlayer: {
				type: 'range',
				min: 10,
				max: 1000,
				step: 5,
				default: this.treatsTotalValuePerPlayer,
				label: 'Total Treat Value Per Player'
			},						
			maxBites: {
				type: 'range',
				min: 1,
				max: 5,
				step: 1,
				default: 3,
				label: 'Max Bites'				
			},
			animalsLimitTotalVisible: {
				type: 'range',
				min: 1,
				max: 12,
				step: 1,
				default: 12,
				label: 'Limit Visible Animals'
			},										
			animalsTotalSnarlish: {
				type: 'range',
				default: 1,
				min: 1,
				max: 20,
				step: 1,
				label: 'Total Snarlish Animals'
			},
			debugAlwaysDisplayTemperaments: {
				type: 'radio',
				default: DEBUG_DISPLAY_ALL_TEMPERAMENTS,
				options: [
					{ value: 0, label: 'No' },
					{ value: 1, label: 'Yes' }
				]
			},						
			debugDisableNFC: {
				type: 'radio',
				default: DEBUG_NFC_DISABLED,
				options: [
					{ value: 0, label: 'No' },
					{ value: 1, label: 'Yes' }
				]
			},			

/*
			playerName: {
				type: 'text',
				default: 'Player 1',
				label: 'Player Name'
			},
			difficulty: {
				type: 'radio',
				default: 'medium',
				options: [
					{ value: 'easy', label: 'Easy' },
					{ value: 'medium', label: 'Medium' },
					{ value: 'hard', label: 'Hard' }
				]
			},
			volume: {
				type: 'range',
				default: 75,
				min: 0,
				max: 100,
				step: 1,
				label: 'Volume Level'
			},
			enableTutorial: {
				type: 'checkbox',
				default: true,
				label: 'Enable Tutorial'
			}
*/				
		};

		// custom styles for the game configuration
		const customStyles = `
				input[type="checkbox"] + label {
					display: inline-flex;
					align-items: center;
					margin-left: 8px;
					transform: translateY(-2px);
				}
		`;
	
		window.config = new GameConfig(configSchema, customStyles);
		document.body.appendChild(window.config.generateForm());
	
		// assigning a global event handler for treatTapped()
		window.treatTapped = (item) => {
            this.treatTapped(item);
        };

		// assigning a global event handler for animalTapped()
		window.animalTapped = (animalID) => {
			this.animalTapped(animalID);
		
		}
		
		// assigning a global event handler for advancePlayer()
		window.advancePlayer = () => {
			console.log('player tapped on the advance player button');
			this.advancePlayer();
		}
		
		// assigning a global event handler for handleNFC()
		window.handleNFC = (message, serialNumber) => {
			this.handleNFC(message, serialNumber);
		}

		this.gameState = GameStates.TitleScreen;
		this.playerState = 0;

		const startElement = this.displayWelcomeScreen();

	/*
		// instantiate the NFC class first as other classes may use it
		this.nfc = new NFC(
			{
				startElement: startElement
			}
		);
	*/

		console.log("this.nfc.active = ",this.nfc.active);

		// DEBUG indicate that NFC became active within 5 seconds
		setTimeout(() => {

			const isActive = this.nfc.isActive();
			console.log("NFC isActive = ",isActive);
	
		}, 5000);

	}

	// the ConfigClass allows us to modify key values
	// here we actually bring them in
	importConfigValues() {

		this.maxPlayers = window.config.maxPlayers;
		//this.treatsPerPlayer = window.config.treatsPerPlayer;
	}

	updatePlayerIndicator(currentPlayer) {

		window.scrollTo({ top: 0, behavior: 'instant' });

		console.log('currentPlayer object = ',currentPlayer);

		if(!this.upperMessageDOM) {

            console.log(`[ERROR] updatePlayerIndicator() doesn't know where in the DOM to display the player name `);
            return false;

        } else {

            const playerNameElement = document.getElementById(this.upperMessageDOM);

            if(!playerNameElement) {

                console.log(`[ERROR] updatePlayerIndicator() couldn't locate a DOM element with id = ${this.upperMessageDOM}`);
                return false;

            } else {

				let playerName = '';

				if(USE_PLAYER_NAMES) {
		
					playerName = `Player ${currentPlayer.number}: ${currentPlayer.name}`;
		
				} else {

					playerName = `Player ${currentPlayer.number}`;

				}
				
				// display the player number and name
				playerNameElement.className = "messageArea";
				playerNameElement.textContent = playerName;
				

			}

		}

	}

	// launched from playerSelectionScreen()
	// handing off an array of names to use for the players
	// order matters

	beginGame(selectedNames) {

		this.totalPlayers = selectedNames.length;

		console.log('totalPlayers = ',this.totalPlayers);

		// instantiate the Treats class
		this.treats = new Treats(
			{
				treatDisplayDOM: this.treatDisplayDOM,
				totalPlayers: this.totalPlayers,
				//treatsPerPlayer: this.treatsPerPlayer, 
				//this.treatsTotalValuePerPlayer: 250
			}
		);

		// instantiate the Animals class
		this.animals = new Animals(
			{
				totalPlayers: this.totalPlayers,
				nfc: this.nfc,
				useNFC: this.nfc.isActive()
			}
		);			

		console.log('totalAnimals = ',this.animals.totalAnimals());

		// instantiate the Players class
		this.players = new Players(
			{
				playerNames: selectedNames,
				//totalPlayers: this.totalPlayers,
				totalAnimals: this.animals.totalAnimals(),
				//maxBites: this.animals.maxBites
				treats: this.treats,
				animals: this.animals
			}
		);			

		// diplay all the treats below for debugging
		//this.treats.displayAllTreats();

		// advance the game state
		this.gameState = GameStates.Running;

		// clear all but the bite container
		this.clearScreen('biteContainer');

		// show the bite indicator until the end of the game
		this.animals.showBites();

		this.advancePlayer();

	}

	turnOffAnimations() {

		// stop new ones from spawing
		this.spawnerId && clearInterval(this.spawnerId);

		// remove any floating hearts
		const heartDivs = document.querySelectorAll('div.heart');
    	heartDivs.forEach(div => div.remove());

		// remove any floating bites
		const biteDivs = document.querySelectorAll('div.floating-bite');
    	biteDivs.forEach(div => div.remove());		

	}

	advancePlayer() {

		this.turnOffAnimations();

		// check for a winner
		const winner = this.players.checkForWinner();
		if(winner) {

			// we have a winner!
			// display the winner screen
			this.displayWinnerScreen(winner);
			return;

		}		

		// check if there are still treats in play
		const treatsRemaining = this.treats.treatsRemaining();
		console.log(`treatsRemaining for all players = ${treatsRemaining}`);

		if(treatsRemaining === false) {

			// when all treats have been used, the game is over
			const winner = this.players.checkForWinner(GameEndingConditions.NoTreatsRemaining);
			this.displayWinnerScreen(winner);
			return;

		}
		
/*		
		// clear the treat display
		const treatDisplay = document.getElementById("playerTreats");
		treatDisplay.innerHTML = '';

		// clear the animal display
		const animalDisplay = document.getElementById("animalSelection");
		animalDisplay.innerHTML = '';
*/

		// clear all but the bite container
		this.clearScreen('biteContainer');

		let currentPlayerObj = {};

		// advance the player and get the current player
		const result = this.players.nextPlayer();
		console.log('advancePlayer() called players.nextPlayer() and return = ',result);		

		// handle getting game ending messages from nextPlayer()
		if(typeof currentPlayerObj === 'object') {
		
			// save the player object we've received
			currentPlayerObj = result;

		} else {	
			
			// handle the case where the game has ended because all players have hit max bites
			if(result == GameEndingConditions.AllPlayersReachedMaxBites) {

				// we'll call checkForWinner to prepare the winner object
				const winner = this.players.checkForWinner();
				this.displayWinnerScreen(winner);
				return;

			}

		}

		console.log('currentPlayer object = ',currentPlayerObj);

		// update the "Player x" indicator and name
		this.updatePlayerIndicator(currentPlayerObj);

		// update the bite counter for this player
		this.animals.setBites(currentPlayerObj.bites);	

		// does the player have more treats?
		const playerHasTreats = this.treats.playerHasTreats(currentPlayerObj.number-1);

		if(!playerHasTreats) {

			// no more treats left for this player
			// logically we should have already skipped this player, 
			// but if we haven't the game will get stuck on an screen 
			// where the player has no treats to select

		} else {

			// display the treats
			this.playerState = PlayerStates.TreatSelection;

			const treatsPrepared = this.treats.preparePlayerTreats(currentPlayerObj.number-1);

			if(treatsPrepared === null) {

				// something went wrong with preparing our treats
				console.log(`advancePlayer() can't display treats`);
		
			} else {

				// update the DOM with the prepared treats
				this.treatsContainer.appendChild(treatsPrepared);
		
			}

		}

	}	

	/*******************************************************************************
	function clearScreen()
	
	Clears all but the specified DIVs within the DOM.
		
	@param  except	a single name of an id to exclude from cleaing

	Returns: nothing
    
    ******************************************************************************/	 
	clearScreen(except = '') {

		const elements = ['upperMessageArea', 'biteContainer', 'playerTreats', 'lowerMessageArea', 'animalSelection'];
		
		elements.forEach(id => {
			if (id !== except) {
				const element = document.getElementById(id);
				if (element) {
					element.innerHTML = '';
				}
			}
		});

		if(except !== 'biteContainer') this.animals.hideBites();
		
	} /* clearScreen() */

    /*******************************************************************************
	function formatWinningMessage()
	
	Prepares a DOM object for the specified animal.
		
	@param  winner        winner object
    
	Returns: text formatted like...
		"1 has won!"	
		"1 & 2 have won!"
		"1, 2, & 3 have won!"

    ******************************************************************************/	 	
	formatWinningMessage(winner) {

		if(winner.draw) {
		
			// it's a draw
			// no players collected any animals

			return "Nobody won.";
		}

		let prefixSingular = '';
		let prefixPlural = '';

		let indexes = winner.winnerIndexes;
		const winnerNames = winner.winnerNames ?? null;

		if(winnerNames) {

			// we'll use names instead of "Player 1"
			// replace the indexes with the actual names
			indexes =  winnerNames;

		} else {

			// not using names, just say "Player 1"
			prefixSingular = 'Player ';
			prefixPlural = 'Players ';
		}

		if (indexes.length === 1) {
			return `${prefixSingular}${indexes[0]} has won!`;
		}
		
		if (indexes.length === 2) {
			return `${prefixSingular}${indexes[0]} & ${indexes[1]} have won!`;
		}
		
		// For 3 or more winners
		const allButLast = indexes.slice(0, -1);
		const last = indexes[indexes.length - 1];
		return `${prefixPlural}${allButLast.join(', ')}, & ${last} have won!`;
	}

	displayWinnerScreen(winner) {

		/* sample data
			{
                tieGame: true,
				draw: false,
                winnerIndexes: [1, 2],
				winnerNames: [  ]
            }
		*/

		if(winner) {

			// TODO clear the DIVs we won't be using
			this.clearScreen('messageArea');			

			// get the element we'll be inserting into
			

			if(!this.upperMessageDOM) {

				console.log(`[ERROR] displayWinnerScreen() doesn't know where in the DOM to display the winner stats `);
				return false;
	
			} else {
	
				const messageArea = document.getElementById(this.upperMessageDOM);
	
				if(!messageArea) {
	
					console.log(`[ERROR] displayWinnerScreen() couldn't locate a DOM element with id = ${this.upperMessageDOM}`);
					return false;
	
				} else {

					const winnerPageWrapper = document.createElement('div');
					winnerPageWrapper.className = 'winner-page';
					//winnerPageWrapper.style.cssText = 'border: 1px solid #ccc; border-radius: 8px; padding: 15px; text-align: center; background-color: white; box-shadow: 0 2px 4px rgba(0,0,0,0.1);';

					// TODO add "play again" option

					console.log("winner object = ",winner);

					const friendOfTotals = this.players.getFriendOfTotals();
					
					//const maximum = Math.max(...friendOfTotals);

					// get indexes of the max value or values
					const max = Math.max(...friendOfTotals);
					const maxIndexes = friendOfTotals
						.map((num, index) => num === max ? index : -1)
						.filter(index => index !== -1);

					console.log("friendOfTotals = ",friendOfTotals);

					let WinnerHeadingText = '';

					// prepare the winner heading text
					if(winner.draw == true) {

						WinnerHeadingText = "It's a draw."

					} else {

						if(winner.tieGame == true) {

							WinnerHeadingText = "We Have a Tie Game!"

						} else {
						
							WinnerHeadingText = "We Have a Winner!"

						}

					}

					const winnerTitle = document.createElement('h1');
					winnerTitle.innerHTML = WinnerHeadingText;
					winnerPageWrapper.appendChild(winnerTitle);

					const winnerDetail = document.createElement('p');				
					const winningText = this.formatWinningMessage(winner);
					winnerDetail.innerHTML = winningText;

					winnerPageWrapper.appendChild(winnerDetail);

					// display totals by player
					const totalsTableWrapper = document.createElement('div');
					totalsTableWrapper.className = 'totals-wrapper';

					const totalsTable = this.players.createTotalResultsTable(friendOfTotals);
					totalsTableWrapper.appendChild(totalsTable);
					winnerPageWrapper.appendChild(totalsTableWrapper);

					// replace any existing content here
					messageArea.replaceChildren(winnerPageWrapper);

					window.scrollTo({ top: 0, left: 0, behavior: 'auto' });  // or smooth

					return;
				}

			}

		}

	}

	displayFriendMessage(animalID = null, possessivePronoun = false) {

		const lowerMessageArea = document.getElementById("lowerMessageArea");
		lowerMessageArea.style.textAlign = 'center';

		const line1 = document.createElement('p');
		line1.innerHTML = `You've made a friend!`;
		line1.style.color = 'purple';
		line1.style.textShadow = '2px 2px 5px, rgb(104, 154, 255)';
		line1.style.fontSize = '2em';
		lowerMessageArea.appendChild(line1);

		console.log(`displayFriendMessage() animalID = ${animalID}`);

		if(animalID) {
			//const pronoun = this.animals.getAnimalPronoun(animalID, possessivePronoun);
			const animalName = this.animals.getAnimalName(animalID);
			const line2 = document.createElement('p');
			line2.style.fontSize = '.8em';
			line2.innerHTML = `Add ${animalName} to your collection.`;
			line2.style.color = 'purple';
			line2.style.textShadow = '2px 2px 5px, rgb(104, 154, 255)';
			line2.style.fontSize = '2em';
			lowerMessageArea.appendChild(line2);
		}

	}

	displayBittenMessage() {

		const lowerMessageArea = document.getElementById("lowerMessageArea");
		lowerMessageArea.style.textAlign = 'center';

		const lowerMessage = document.createElement('span');
		lowerMessage.innerHTML = `You've been bitten!`;
		lowerMessage.style.color = 'red';
		lowerMessage.style.textShadow = '2px 2px 5px #ffdb68';
		lowerMessage.style.fontSize = '2em';

		lowerMessageArea.appendChild(lowerMessage);

	}

	// DEPRECIATED
	displayWelcomeScreenWithPlayerNumbers() {

		if(!this.upperMessageDOM) {

            console.log(`[ERROR] displayWelcomeScreen() doesn't know where in the DOM to display the welcome message `);
            return false;

        } else {

			const startElement = document.getElementById(this.upperMessageDOM);

            if(!startElement) {

                console.log(`[ERROR] displayWelcomeScreen() couldn't locate a DOM element with id = ${this.upperMessageDOM}`);
                return false;

            } else {

				const startPageWrapper = document.createElement('div');
				startPageWrapper.className = 'start-page';
				startPageWrapper.style.cssText = `
					border: 1px solid #ccc;
					border-radius: 8px;
					padding: 15px;
					padding-bottom: 45px;
					text-align: center;
					background-color: white;
					box-shadow: 0 2px 4px rgba(0,0,0,0.1);
				`;
				
				const gameTitle = document.createElement('h1');
				gameTitle.innerHTML = "Animal Friends and Enemies";
				startPageWrapper.appendChild(gameTitle);

				// create several buttons up to this.maxPlayers
				for (let i = 0; i < this.maxPlayers; i++) {

					let button = document.createElement('div');
					button.className = 'selectPlayers';
					button.style.cssText = `
						width: 65%;
						height: 1.3em;
						margin: 0 auto;
						margin-top: 13px;
						background-color:rgb(104, 98, 199);
						color: white;
						display: flex;
						align-items: center;
						justify-content: center;
						cursor: pointer;
						user-select: none;
						border-radius: 15px;
						box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
					}
					`;
					
					// handle plurals
					if(i == 0) { 

						button.innerHTML = `${i+1} Player`;

					} else {

						button.innerHTML = `${i+1} Players`;

					}					

					button.addEventListener("click", (event) => {
						//event.preventDefault(); // prevent the default link behavior
						//this.handleTreatTap(item);
						this.beginGame(i+1);
					}, { once: true });

					startPageWrapper.appendChild(button);

				}
				
				startElement.appendChild(startPageWrapper);
		
				window.scrollTo({ top: 0, left: 0, behavior: 'auto' });  // or smooth
		
				return startElement;

			}

		}

	}

	playerSelectionScreen() {

		this.importConfigValues();

		// clear the welcome screen
		this.clearScreen('biteContainer');
		
		// scroll to the top
		window.scrollTo({ top: 0, left: 0, behavior: 'auto' });

		// name selector component
		const selector = new NameSelector({
			names: [
                'Kristi', 'KT', 'Wes', 'Hillary',
                'Mom', 'Hermoine', 'Lou', 'Karen',
                'Emmy', 'Keis', 'Nel', 'Parker', 
                'Kelli', 'Jake', 'Chloe', 'Lincoln', 'Lucy',
                'Bolt', 'Frost', 'Nova',
                'Spark', 'Blitz', 'Flux',
                'Zephyr', 'Ember', 'Nyx',
                'Solara', 'Glitch', 
                'Pulse', 'Cipher', 'Fizz',
                'Snap', 'Skye', 'Zap'
            ],   
			minPlayers: 1,
			maxPlayers: this.maxPlayers,
			title: 'Select Your Names'
		});

		// Show the selector in a specific element
		const targetElement = document.getElementById(this.upperMessageDOM);
		const startGame = async () => {
			const selectedNames = await selector.show(targetElement);
			console.log('Selected players:', selectedNames);
			this.beginGame(selectedNames);
		};

		startGame();

	}

	async displayWelcomeScreen() {

		const startElement = document.getElementById(this.upperMessageDOM);

		const startPageWrapper = document.createElement('div');
		startPageWrapper.className = 'start-page';
		startPageWrapper.style.cssText = `
			border: 1px solid #ccc;
			border-radius: 8px;
			padding: 15px;
			padding-bottom: 45px;
			text-align: center;
			background-color: white;
			box-shadow: 0 2px 4px rgba(0,0,0,0.1);
		`;

		const gameTitle = document.createElement('h1');
		gameTitle.innerHTML = "Animal Friends and Enemies";
		startPageWrapper.appendChild(gameTitle);

		let button = document.createElement('div');
		button.className = 'selectPlayers';
		button.style.cssText = `
			font-size: .9em;
			padding: 8px;
			width: 5em;
			height: 1.3em;
			margin: 0 auto;
			margin-top: 13px;
			background-color:rgb(104, 98, 199);
			color: white;
			display: flex;
			align-items: center;
			justify-content: center;
			cursor: pointer;
			user-select: none;
			border-radius: 15px;
			box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
		`;
		button.innerHTML = 'Begin';

		button.addEventListener("click", async (event) => {
			await this.playerSelectionScreen();
		}, { once: true });

		startPageWrapper.appendChild(button);
		startElement.appendChild(startPageWrapper);

		// instantiate the NFC class early as other classes may use it
		this.nfc = new NFC(
			{
				//startElement: startElement
				startElementID: this.upperMessageDOM
			}
		);

		//await window.scrollTo({ top: 0, left: 0, behavior: 'auto' });
		
		return;
	}

	// DEPRECIATED
	displayWelcomeScreenClassic() {

		if(!this.upperMessageDOM) {

            console.log(`[ERROR] displayWelcomeScreen() doesn't know where in the DOM to display the welcome message `);
            return false;

        } else {

			const startElement = document.getElementById(this.upperMessageDOM);

            if(!startElement) {

                console.log(`[ERROR] displayWelcomeScreen() couldn't locate a DOM element with id = ${this.upperMessageDOM}`);
                return false;

            } else {

				const startPageWrapper = document.createElement('div');
				startPageWrapper.className = 'start-page';
				startPageWrapper.style.cssText = `
					border: 1px solid #ccc;
					border-radius: 8px;
					padding: 15px;
					padding-bottom: 45px;
					text-align: center;
					background-color: white;
					box-shadow: 0 2px 4px rgba(0,0,0,0.1);
				`;
				
				const gameTitle = document.createElement('h1');
				gameTitle.innerHTML = "Animal Friends and Enemies";
				startPageWrapper.appendChild(gameTitle);

/*
					button.addEventListener("click", (event) => {
						//event.preventDefault(); // prevent the default link behavior
						//this.handleTreatTap(item);
						this.beginGame(i+1);
					}, { once: true });
*/
				
				startElement.appendChild(startPageWrapper);
		
				window.scrollTo({ top: 0, left: 0, behavior: 'auto' });  // or smooth
		
				return startElement;

			}

		}

	}

	showExtendedInfo() {

		// only show if flag set
		if (model.showExtendedInfo === true) {

			let treatExtendedDiv = document.getElementById("extended-info");
			treatExtendedDiv.innerHTML = '<h3>Players</h3><pre>' + JSON.stringify(model.players, null, 2) + '</pre>' +
				'<h3>Animals</h3><pre>' + JSON.stringify(model.animals, null, 2) + '</pre>';

		}

	}

	treatTapped(item) {

		//alert("Clicked: " + JSON.stringify(item));

		// validate that this event is happening at the correct stage
		if(this.gameState == GameStates.Running && 
		   	this.playerState == PlayerStates.TreatSelection) {

			// advance the player to Animal Selection
			this.playerState = PlayerStates.AnimalSelection;

			const currentPlayerObj = this.players.getCurrentPlayerObj();
			
			// pass this event on to our treat class
			const success = this.treats.handleTreatTap(item,currentPlayerObj);

			if(!success) {

				// failed to properly handle treats
				// do nothing

			} else {

				console.log("this.nfc.active = ",this.nfc.active);

				// if NFC isn't active, show the available animals
				if(this.nfc.active === true) {
	
					console.log("NFC is active so just wait for an animal to be scanned")
	
				} else {
	
					this.animals.displayAvailableAnimals(currentPlayerObj.number-1);
	
				}

			}

		} else {

			// player tapped on an treat when it wasn't expected
			// this really shouldn't happen during normal game play

			const stateDebugMessage = this.prepareStateDebugMessage();

			console.log("treatTapped() player tapped on a treat when it wasn't expected, here's the currrent state: ",stateDebugMessage);

		}

	}

	prepareStateDebugMessage() {

		let msg = '';
	
		// prepare the game state message
		msg += `Game State: (${this.gameState}) ${GameStateMessages[this.gameState]} ||| `;

		// prepare the player state message
		msg += `Player State: (${this.playerState}) ${PlayerStateMessages[this.playerState]}`;

		// add the current player info
		const currentPlayerObj = this.players.getCurrentPlayerObj();
		
		if(currentPlayerObj) {

			msg += ` ||| Current Player: Player ${currentPlayerObj.number}`;

			// conditionally add the current player's name too
			if(USE_PLAYER_NAMES) {
				
				msg += `: ${currentPlayerObj.name} `;

			}

		}

		return msg;

	}
	

	/*******************************************************************************
     * @function animalTapped
     *
     * @description Handles taps for animals cards and those triggered by NFC taps
     * 
     * @details
     * - First, the game state is checked to ensure that user taps or NFC taps
	 *   aren't calling this function when we aren't expecting it.
	 * - Advances the player state to PlayerStates.AnimalStats
	 * - Prepares the animal stats and interaction card.
     * 
     * @param {integer} animalID - an index of one of the animals
     * @returns {void} 
     * 
     * @example
     * animalTapped(animalID);
	 * 
	 * @see animalTappedByType, handleNFC
     * 
     * @author Kristi
     * @since 2025-01-05
     * @lastModified 2025-01-05
     ******************************************************************************/

	animalTapped(animalID) {

		//alert("Clicked: " + JSON.stringify(animalID));

		// validate that this event is happening at the correct stage
		if(this.gameState == GameStates.Running && 
		   	this.playerState == PlayerStates.AnimalSelection) {

			// handle the interaction between the player and the selected animal
			const currentPlayerObj = this.players.getCurrentPlayerObj();
			const currentTreatObj = this.treats.getCurrentTreatObj();
			console.log(`animalTapped() currentTreatObj = `,currentTreatObj);
			const interaction = this.animals.handleAnimalInteraction(animalID, currentPlayerObj, currentTreatObj);

			if(interaction == AnimalInteractions.NewBite) {

				// update this player's bite count
				const playerHasMaxBites = this.players.playerBitten();
				console.log(`animalTapped() playerHasMaxBites = `,playerHasMaxBites);
			
				// also update the bite display
				this.animals.setBites(currentPlayerObj.bites);

				this.displayBittenMessage();

				// create floating images
				const spawnInterval = 200;
				this.spawnerId = setInterval(this.animals.createFloatingImage, spawnInterval);
				// I'll turn off later with: clearInterval(this.spawnerId);

				// if the player has hit max bites, redistribute their treats to other players
				if(playerHasMaxBites) {

					// save the current player's index as the player to lose their treats
					const playerIndexLosingTreats = currentPlayerObj.number-1;
					console.log('playerIndexLosingTreats = ',playerIndexLosingTreats);

					// prepare an array of player indexes who should receive the lost treats
					const playersGainingTreatsArray = this.treats.playersWithTreatsRemaining(playerIndexLosingTreats);
					console.log('playersGainingTreatsArray = ',playersGainingTreatsArray);

					console.log('treats before redistribution',this.treats.treats);

					// if playersGainingTreatsArray isn't empty, then redistribute the lost treats
					if(playersGainingTreatsArray.length !== 0) {

						// redistribute the lost treats randomly to the remaining players
						this.treats.redistributePlayerTreats(playerIndexLosingTreats, playersGainingTreatsArray);

					}

					console.log('treats after redistribution',this.treats.treats);

				}

			}
				
			if(interaction == AnimalInteractions.NewFriend) {

				// increment this player's friend counter
				this.players.playerFriended();

				// display the new friend message
				this.displayFriendMessage(animalID, true);

				// display the floating hearts
				this.spawnerId = setInterval(this.animals.createHeart, 300);


			}			
		
			// advance the player to Animal Stats
			this.playerState = PlayerStates.AnimalStats;

			// clear the treat display
			const treatDisplay = document.getElementById("playerTreats");
			treatDisplay.innerHTML = '';

			// get the animal stats card and then display it
			const animalCard = this.animals.prepareAnimalStats(animalID, interaction);
			const animalStats = document.getElementById("animalSelection");
			animalStats.replaceChildren(animalCard);
			//animalCard.scrollIntoView({ behavior: 'smooth' });
			window.scrollTo({ top: 0, behavior: 'smooth' });

			// display this animal's temperament graph
			this.animals.addAnimalTemperamentGraphByID(animalID,currentPlayerObj.number-1);
			console.log('current player = ',this.players.currentPlayer);

		} else {

			const stateDebugMessage = this.prepareStateDebugMessage();

			console.log("animalTapped() player tapped on an animal when it wasn't expected, here's the currrent state: ",stateDebugMessage);

		}

	}

	animalTappedByType(animalType) {

		const animalID = this.animals.matchAnimalTypeToID(animalType);

		this.animalTapped(animalID);

	}

	handleNFC(message, serialNumber) {
	
		console.log(`NFC Read: (${message.records.length})`);

		// get data stored about the tag
		const tag = this.nfc.getTagBySerialNumber(serialNumber);

		if(!tag) {

			console.log(`handleNFC() scanned a tag but found no matching data for ID: ${serialNumber}`);

			return;
 
		} else {

			console.log("handleNFC() tag data found = ",tag);

			// validate that this is the proper collection for this game

			if(tag.collection == "animals") {

				console.log(`handleNFC() working with a tag from this game's collection`);

				this.animalTappedByType(tag.type);

				return;

			}

			console.log(`handleNFC() scanned a tag but it's from another collection = ${tag.collection}`);

		}
	  
	}
	
};

//let game = null;

window.onload = init;

function init() {

	window.game = new AnimalFriendsGame();
	
	return;


	nfc = new NFC();

	console.log("this.nfc.active = ",this.nfc.active);

	setTimeout(() => {

		const isActive = nfc.isActive();
		console.log("isActive = ",isActive);

		console.log("this.nfc.active = ",this.nfc.active);

	}, 5000);

	return;

	const tag = nfc.getTagBySerialNumber("04:b3:6b:62:4a:6e:80");
	console.log("getTagBySerialNumber()",tag);

	const tags = nfc.getTagByType("treasure");
	console.log("getTagByType()", tags);

	const collection = nfc.getTagCollection("animals");
	console.log("getTagCollection()", collection);

	return;

	animals.displayAllAnimals();

	return;

/*	
	// show just 1 animal card
	animal = animals.prepareAnimalCard(1, true);

	document.body.appendChild(animal);

	return;
*/


	let treat;

	treat = treats.getTreatByIndex(0);
	console.log(treat);

	treat = treats.getTreatByIndex(2);
	console.log(treat);

	treat = treats.getTreatByIndex(1);
	console.log(treat);	

	treat = treats.getTreatByIndex(4);
	console.log(treat);		

	treat = treats.getTreatByIndex(0);
	console.log(treat);	

	treat = treats.getTreatByIndex(0);
	console.log(treat);	

	treat = treats.getTreatByIndex(0);
	console.log(treat);	

	treat = treats.getTreatByIndex(0);
	console.log(treat);		


/*	
	while(treat = treats.getTreatByIndex(1) !== null ) {	

		console.log(treat);

	}
*/

/*	
	while(treat = treats.getRandomTreat(1) !== null ) {

		console.log(treat);

	}
*/	
	

	// DEBUG validate treat values function
	//let myTreats = treats.validateGenerateTreatValues(5,250,1000);
	
	//let myTreats = treats.selectTreats(3,5,250);

	// JSON.stringify
	//console.log(myTreats);

	// here we show only player 1's treats
	//console.log(myTreats[0]);

	//model.selectTreats();
	
	//view.showExtendedInfo();

	//console.log(model.players);	

}