import { 
	USE_PLAYER_NAMES,
    GameEndingConditions,
    GameEndingMessages
} from './GameConstants.js';

export class Players {

    currentPlayer = null;
    totalPlayers = 0;
    totalAnimals = null;

    // a reference to our treats class
    treats = null; 

    // a reference to our animals class
    animals = null;

    // values imported from config
    maxBites = null;

    players = [
        { number: 1, color: null, animals: 0, bites: 0, turns: 0, name: "Kristi" },
        { number: 2, color: null, animals: 0, bites: 0, turns: 0, name: "Bunny" },
        { number: 3, color: null, animals: 0, bites: 0, turns: 0, name: "Fox" },
        { number: 4, color: null, animals: 0, bites: 0, turns: 0, name: "Big Kitty" }
    ];

    constructor(parms) {

        //console.log('Players Class Instantiated');

        //this.currentPlayer = 1;

        // save a reference to our classes
        this.treats = parms.treats;
        this.animals = parms.animals;

        // import defaults from the config object
        this.maxBites = window.config.maxBites;

        const newPlayerNames = parms.playerNames;
        this.players = this.updatePlayerNames(this.players, newPlayerNames);

        this.totalPlayers = newPlayerNames.length;

        /*
        // slice the extra players out of the array
        if (this.totalPlayers < this.players.length) {
            this.players = this.players.slice(0, this.totalPlayers);
        }
*/

        this.totalAnimals = parms.totalAnimals;
        //this.maxBites = parms.maxBites;

        // use the maxBites value from the global config
        this.maxBites = window.config.maxBites;

    }

    updatePlayerNames(players, newNames) {
        return players
            .slice(0, newNames.length)
            .map((player, index) => ({ ...player, name: newNames[index] }));
    }


    nextPlayer() {


        if (this.currentPlayer === null) {

            // the game is beginning,
            // start with the first player
            this.currentPlayer = 0;
            return this.players[this.currentPlayer];

        } else {

            let nextPlayer = this.currentPlayer;
            let skippedPlayers = 0;
            let validPlayer = false;
            
            // rather than a traditional for loop or iterating through the players
            // we loop through all players starting with the current player
            do {

                // here we advance to the next player
                nextPlayer = (nextPlayer + 1) % this.totalPlayers;
                skippedPlayers++;

                // set the validPlayer flag to true before we do any checks
                validPlayer = true;

                // check if this player has hit maxBites
                if (this.players[nextPlayer].bites == this.maxBites) {

                    validPlayer = false;

                }

                // check if this player still has treats remaining
                if (!this.treats.playerHasTreats(nextPlayer)) {

                    validPlayer = false;

                }

                // if all checks have passed, we have a valid player
                if (validPlayer) {

                    this.currentPlayer = nextPlayer;
                    return this.players[this.currentPlayer];

                }
                
                // check if we've checked all players
                if (skippedPlayers >= this.totalPlayers) {

                    return GameEndingConditions.NoPlayersCanTakeATurn;

                }

            } while (true);

        }

    }
    
    /*******************************************************************************
	function checkForWinner()
	
	Checks if anyone has won the game
		
	Returns: an array with details about the winner or winners

    ******************************************************************************/	        
    checkForWinner(gameEndingCondition = false) {

        let winner = false;

        // handle any game ending conditions
        if(gameEndingCondition) {

            const gameEndingConditionMessage = GameEndingMessages[gameEndingCondition];
            console.log(`checkForWinner() game has ended because ${gameEndingConditionMessage}`);

            winner = true;

        } else {

            console.log('checkForWinner() checking for winner');

        }

        const allPlayersExceededBites = Object.values(this.players).every(player => 
            player.bites >= this.maxBites
        );

        if(allPlayersExceededBites) {

            winner = true;

        }

        // calculate total animals across all players
        const currentTotalAnimals = Object.values(this.players).reduce((sum, player) => 
            sum + player.animals, 0
        );

        console.log(`checkForWinner() total animals collected = ${currentTotalAnimals}, totalAnimals = ${this.totalAnimals}`);

        // have all of the animals been collected?
        const animalsRemaining = this.animals.totalAnimalsRemaining();
        console.log(`checkForWinner() animalsRemaining = ${animalsRemaining}`);

        if(animalsRemaining == 0) {

            winner = true;

        }

        if(winner) {

            console.log('checkForWinner() we have a winner!');

            // we have one or more winners!
            const maxAnimalPlayers = this.getMaxAnimalPlayers();
            console.log('checkForWinner() maxAnimalPlayers = ',maxAnimalPlayers);

            /*
            maxAnimalPlayers = {
                maxPlayerIndexes,
                maxPlayerNames,
                tieGame
            };
            */

            // do we have a tie game
            const tieGame = maxAnimalPlayers > 1 ?? false;

            // check for a draw
            // which can only be if no players collected any animals
            const draw = currentTotalAnimals == 0 ?? false;

            // DEBUG always return a winner
            let winnerObject = {
                tieGame: maxAnimalPlayers.tieGame,
                draw: draw,
                winnerIndexes: maxAnimalPlayers.maxPlayerIndexes,
            };

            // if USE_PLAYER_NAMES, otherwise leave off
            if(USE_PLAYER_NAMES) winnerObject.winnerNames = maxAnimalPlayers.maxPlayerNames;

            return winnerObject;

        }

        console.log('checkForWinner() no winner yet');

        return null;

    } /* end checkForWinner() */

    /*******************************************************************************
	function getMaxAnimalPlayers()
	
    Prepares an array with the indexes of the players who have collected the 
    most animals.

    Normally, there is only one winner, but a tie is also possible.
	
    @param  none

	Returns: an array with the player indexes 

    ******************************************************************************/	          
    getMaxAnimalPlayers(players) {
        // Find the maximum number of animals
        const maxAnimals = Math.max(...Object.values(this.players).map(player => player.animals));
        
        // Find all players with that maximum
        const maxPlayerIndexes = [];
        const maxPlayerNames = [];
        
        for (let i = 0; i <= Object.keys(this.players).length-1; i++) {
            if (this.players[i].animals === maxAnimals) {
                maxPlayerIndexes.push(i);
                maxPlayerNames.push(this.players[i].name);
            }
        }
        
        const tieGame = maxPlayerIndexes.length > 1 ?? false;
        
        return {
            maxPlayerIndexes,
            maxPlayerNames,
            tieGame
        };

    } /* end getMaxAnimalPlayers */

    getCurrentPlayerObj() {

        return this.players[this.currentPlayer];

    }

    // add a bite for this player
    playerBitten() {

        const currentPlayerIndex = this.currentPlayer;

		this.players[currentPlayerIndex].bites++;
        
        // has the player hit max bites?
        if(this.players[currentPlayerIndex].bites == this.maxBites) {

            return true;

        }

        return false;

    }

    // add a bite for this player
    playerFriended() {

        const currentPlayerIndex = this.currentPlayer;

		this.players[currentPlayerIndex].animals++;

    }    

    bites() {

        const currentPlayerIndex = this.currentPlayer;

		const playerBites = this.players[currentPlayerIndex].bites;

        return playerBites;

    }

    getFriendOfTotals() {

        const totals = [];
        //totals[0] = null; // To maintain 1-based indexing
        
        for (let i = 0; i <= Object.keys(this.players).length-1; i++) {
            totals[i] = this.players[i].animals;
        }
        
        return totals;

    }

    /*******************************************************************************
	function createTotalResultsTable()
	
    Prepares a DOM object containing a table of players and the number
    of animals that each has collected.
	
    @param  totals  the output from 
        const friendOfTotals = this.getFriendOfTotals();
		const friendOfTotals = [0, 4, 5, 2, 3];

	Returns: a DOM object

    ******************************************************************************/	        
    createTotalResultsTable(totals) {

        const totalTable = document.createElement('table');
        totalTable.className = 'totals-table'; // for styling
        const style = document.createElement('style');

        // Create header row
        const headerRow = document.createElement('tr');
        const playerHeader = document.createElement('th');
        playerHeader.textContent = 'Player';
        const animalsHeader = document.createElement('th');
        animalsHeader.textContent = 'Animals';
        headerRow.appendChild(playerHeader);
        headerRow.appendChild(animalsHeader);
        totalTable.appendChild(headerRow);

        // Create rows for each player, starting from index 1 to skip placeholder
        for (let player = 0; player < totals.length; player++) {
            const row = document.createElement('tr');
            
            const playerCell = document.createElement('td');

            // use Player 1 or Name
            if(USE_PLAYER_NAMES) {
                
                console.log('player = ',player);
                playerCell.textContent = this.players[player].name;
    
            } else {

                playerCell.textContent = `Player ${player}`;

            }
            
            const totalCell = document.createElement('td');
            totalCell.textContent = totals[player];
            
            row.appendChild(playerCell);
            row.appendChild(totalCell);
            totalTable.appendChild(row);
        }
    
        return totalTable;
    }







}